﻿using System.Runtime.CompilerServices;
using ProChain.Api.Base;
using ProChain.Api.Fusion;
using ProChain.App.Base.Dto.Auth;
using ProChain.App.Fusion.Dto.Filters;
using ProChain.App.Fusion.Dto.Tasks;
using ProChain.App.Fusion.Dto.Users;

namespace ProChain.Fusion.Sdk.Examples
{
    public class ProChainFusionApiExample
    {
        private readonly IProChainFusionApiClient _client;

        private readonly string _serverUrl;

        private readonly string _username;

        private readonly string _password;

        public ProChainFusionApiExample(string serverUrl, string username, string password)
        {
            _serverUrl = serverUrl;

            _username = username;

            _password = password;

            _client = ProChainWebFusionClientFactory.Create(
                localStorage: new InMemoryStorage(),
                onLoginRequired: OnLoginRequired,
                serverUrl: _serverUrl);
        }

        private async Task<bool> OnLoginRequired(IProChainFusionApiClient apiClient)
        {
            //return true to repeat the request
            return await Login();
        }

        public async Task TestWorkflow()
        {
            if (!await IsAvailable())
            {
                return;
            }

            if (!await ShowSystemInfo())
            {
                return;
            }

            //skip this step to test OnLoginRequired flow
            if (!await Login())
            {
                return;
            }

            var userLoginInfo = await ShowUserInformation();

            await ShowMyTasks(userLoginInfo);

            //Logout invalidates a refresh token on the server side
            await Logout();
        }

        private async Task<bool> IsAvailable()
        {
            AddHeader();

            var apiCall = await _client.SystemService.IsAvailable().ToApiCallResult();

            if (ShowError(apiCall))
            {
                return false;
            }

            var isAvailable = apiCall.Result;

            Console.WriteLine($"Access to {_serverUrl} is available: [{isAvailable}]");

            return isAvailable;
        }

        private async Task<bool> ShowSystemInfo()
        {
            AddHeader();

            var apiCall = await _client.SystemService.GetSystemInfo().ToApiCallResult();

            if (ShowError(apiCall))
            {
                return false;
            }

            var systemInfo = apiCall.Result;

            var isTenantActive = systemInfo.IsTenantActive;

            if (isTenantActive)
            {
                Console.WriteLine($"{_serverUrl} is active.");

                Console.WriteLine($"API version: [{systemInfo.AppVersion}]");

                Console.WriteLine($"Company name: [{systemInfo.CompanyName}]");

                Console.WriteLine($"Support email: [{systemInfo.SupportEmail}]");
            }
            else
            {
                Console.WriteLine($"{_serverUrl} is not active.");
            }

            return isTenantActive;
        }

        private async Task<bool> Login()
        {
            AddHeader();

            var apiCall = await _client.Login(_username, _password).ToApiCallResult();

            if (ShowError(apiCall))
            {
                return false;
            }

            var loginResult = apiCall.Result;

            var isSuccess = loginResult.IsSuccess();
            if (!isSuccess)
            {
                Console.WriteLine($"Failed to login [{loginResult}]");

                return false;
            }

            return true;
        }

        private async Task Logout()
        {
            AddHeader();

            var apiCall = await _client.Logout().ToApiCallResult();

            ShowError(apiCall);
        }

        private async Task<UserLoginInfoDto?> ShowUserInformation()
        {
            AddHeader();

            var apiCall = await _client.SessionService.GetCurrentLoginInfo().ToApiCallResult();

            if (ShowError(apiCall))
            {
                return null;
            }

            var sessionInfo = apiCall.Result;

            Console.WriteLine($"User Id:  {sessionInfo.CurrentUser.Id}");
            Console.WriteLine($"Username:  {sessionInfo.CurrentUser.Username}");
            Console.WriteLine($"FirstName:  {sessionInfo.CurrentUser.FirstName}");
            Console.WriteLine($"LastName:  {sessionInfo.CurrentUser.LastName}");

            Console.WriteLine($"SystemAdministrator:  {sessionInfo.CurrentUser.IsSystemAdministrator}");

            Console.WriteLine();

            return sessionInfo.CurrentUser;
        }

        private async Task ShowMyTasks(UserProfileDto? user)
        {
            if (user == null)
            {
                return;
            }

            AddHeader();

            var apiCall = await _client.TaskService.SearchTasks(new TaskSearchOptionsDto
            {
                Skip = 0,
                MaxResults = 25,
                Sorts = new List<TaskViewSort>
                {
                    TaskViewSort.AssignmentScheduleOrder
                },
                DataFields = new List<TaskDetailsField>
                {
                    TaskDetailsField.AssignedUser,
                    TaskDetailsField.Indicators,
                    TaskDetailsField.AssignmentDates
                },
                Filters = new TaskFiltersDto
                {
                    AssignmentStart = new AssignmentStartFilter
                    {
                        Horizon = HorizonType.TwoWeeks,
                        IncludeCompletedTasks = false
                    },
                    AssignedUsers = new List<FilterItem>
                    {
                        new FilterItem
                        {
                            Id = user.Id
                        }
                    }
                }
            }).ToApiCallResult();

            if (ShowError(apiCall))
            {
                return;
            }

            var pagedResult = apiCall.Result;

            if (pagedResult.Items.Count > 0)
            {
                Console.WriteLine($"You have [{pagedResult.Items.Count}] assigned tasks:");

                foreach (var taskDetails in pagedResult.Items)
                {
                    Console.WriteLine(taskDetails.Name);
                }
            }
            else
            {
                Console.WriteLine($"All done. There are no assigned tasks to you.");
            }

            Console.WriteLine();
        }

        private static bool ShowError(ApiCallResult apiCallResult)
        {
            if (apiCallResult.IsSuccessful)
            {
                return false;
            }

            Console.WriteLine($"Error reason [{apiCallResult.ExceptionType}]. {apiCallResult.Exception.Message}");

            return true;
        }

        private static void AddHeader([CallerMemberName] string? name = null)
        {
            Console.WriteLine();
            Console.WriteLine($"---  {name}  ---");
            Console.WriteLine();
        }
    }
}